"""
https://github.com/fbiego/BLE_OTA_Python
  
MIT License

Copyright (c) 2021 Felix Biego

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in all
copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
SOFTWARE.
"""

from __future__ import print_function
from os import path
import asyncio
import math
import sys
import re
import hashlib

from bleak import BleakClient, BleakScanner
from bleak.exc import BleakError

header = """#####################################################################
    ----------------- TailCoNTROL BLE OTA update ----------------
#####################################################################"""

UART_SERVICE_UUID = ""
UART_RX_CHAR_UUID = ""

# UUIDs for Device Information Service (DIS) characteristics
DIS_CHARACTERISTICS = {
    "manufacturer_name": "00002a29-0000-1000-8000-00805f9b34fb",
    "model_number": "00002a24-0000-1000-8000-00805f9b34fb",
    "serial_number": "00002a25-0000-1000-8000-00805f9b34fb",
    "hardware_revision": "00002a27-0000-1000-8000-00805f9b34fb",
    "firmware_revision": "00002a26-0000-1000-8000-00805f9b34fb",
    "software_revision": "00002a28-0000-1000-8000-00805f9b34fb",
}

PART = 16000
MTU = 500

clt = None
fileBytes = None
total = 0
curposition = 0


async def start_ota(ble_address: str, file_name: str):
    def handle_disconnect(_: BleakClient):
        global disconnect
        disconnect = False
        end = False
        print("\nDevice disconnected")
        disconnected_event.set()

    async def handle_rx(_: int, data: bytearray):
        global curposition
        toSend = bytearray()
        for x in range(0, MTU):
            if (curposition+x >= len(fileBytes)):
                break
            toSend.append(fileBytes[curposition+x])
        curposition = curposition + MTU
        await send_data(client, toSend, False)
        printProgressBar(curposition + x + 1, total,
                         prefix='Progress:', suffix='Complete', length=50)

    def printProgressBar(iteration, total, prefix='', suffix='', decimals=1, length=100, fill='█', printEnd="\r"):
        """
        Call in a loop to create terminal progress bar
        @params:
            iteration   - Required  : current iteration (Int)
            total       - Required  : total iterations (Int)
            prefix      - Optional  : prefix string (Str)
            suffix      - Optional  : suffix string (Str)
            decimals    - Optional  : positive number of decimals in percent complete (Int)
            length      - Optional  : character length of bar (Int)
            fill        - Optional  : bar fill character (Str)
            printEnd    - Optional  : end character (e.g. "\r", "\r\n") (Str)
        """
        percent = ("{0:." + str(decimals) + "f}").format(100 *
                                                         (iteration / float(total)) - 0.1)
        filledLength = int(length * iteration // total)
        bar = fill * filledLength + '-' * (length - filledLength)
        print(f'\r{prefix} |{bar}| {percent}% {suffix}', end=printEnd)
        # Print New Line on Complete
        if iteration == total:
            print()

    async def send_data(client: BleakClient, data: bytearray, response: bool):
        if client.__class__.__name__ == "BleakClientBlueZDBus":
            await client._acquire_mtu()

        # Write without response is limited to MTU - 3 bytes
        chunk_size = client.mtu_size - 3
        for chunk in (data[i: i + chunk_size] for i in range(0, len(data), chunk_size)):
            await client.write_gatt_char(UART_RX_CHAR_UUID, chunk, response)

    device = await BleakScanner.find_device_by_address(ble_address, timeout=20.0)
    disconnected_event = asyncio.Event()

    if not device:
        print("------------ Failed ------------")
        print(f"Device with address {ble_address} could not be found.")
        return

    print(f"Device Name: {device.name}")
    dis_present = False

    try:
        async with BleakClient(device, disconnected_callback=handle_disconnect) as client:
            print(f"Connected: {client.is_connected}")

            for name, uuid in DIS_CHARACTERISTICS.items():
                try:
                    value = await client.read_gatt_char(uuid)
                    print(
                        f"{name.replace('_', ' ').title()}: {value.decode('utf-8')}")
                    dis_present = True
                except Exception:
                    pass

            if dis_present:
                print("TailCoNTROL UUIDS set")
                # TailCoNTROL BLE Service UUIDs
                UART_SERVICE_UUID = "19F8ADE2-D0C6-4C0A-912A-30601D9B3060"
                UART_RX_CHAR_UUID = "5E4D86AC-EF2F-466F-A857-8776D45FFBC2"
            elif device.name == "EG2":
                print("Legacy EG2 UUIDS set")
                # EG2 BLE Service UUIDs
                UART_SERVICE_UUID = "927dee04-ddd4-4582-8e42-69dc9fbfae66"
                UART_RX_CHAR_UUID = "05e026d8-b395-4416-9f8a-c00d6c3781b9"
            else:
                print("Legacy MiTail UUIDS set")
                # MiTail BLE Service UUIDs
                UART_SERVICE_UUID = "3af2108b-d066-42da-a7d4-55648fa0a9b6"
                UART_RX_CHAR_UUID = "5bfd6484-ddee-4723-bfe6-b653372bbfd6"

            await client.start_notify(UART_RX_CHAR_UUID, handle_rx)
            await asyncio.sleep(1.0)

            def get_bytes_from_file(filename):
                print("Reading from: ", filename)
                return open(filename, "rb").read()

            global fileBytes
            fileBytes = get_bytes_from_file(file_name)
            global clt
            global curposition
            clt = client
            fileParts = math.ceil(len(fileBytes) / PART)
            global total
            total = len(fileBytes)
            otaInfo = bytes("OTA " + str(len(fileBytes)) + " " +
                            str(hashlib.md5(fileBytes).hexdigest()), 'utf-8')

            await send_data(client, otaInfo, True)

            await disconnected_event.wait()
            print("----------- Complete -----------")
    except BleakError as e:
        print(f"Failed to connect to device: {e}")


def isValidAddress(str):
    # Regex to check valid MAC address
    regex = ("^([0-9A-Fa-f]{2}[:-])" +
             "{5}([0-9A-Fa-f]{2})|" +
             "([0-9a-fA-F]{4}\\." +
             "[0-9a-fA-F]{4}\\." +
             "[0-9a-fA-F]{4}){17}$")
    regex2 = "^[{]?[0-9a-fA-F]{8}" + \
        "-([0-9a-fA-F]{4}-)" + "{3}[0-9a-fA-F]{12}[}]?$"

    # Compile the RegEx
    p = re.compile(regex)
    q = re.compile(regex2)

    # If the string is empty return false
    if (str == None):
        return False

    # Return if the string matched the RegEx
    if (re.search(p, str) and len(str) == 17):
        return True
    else:
        if (re.search(q, str) and len(str) == 36):
            return True
        else:
            return False


if __name__ == "__main__":
    print(header)
    if (len(sys.argv) > 2):
        if isValidAddress(sys.argv[1]) and path.exists(sys.argv[2]):
            print("Trying to start OTA update...")
            asyncio.run(start_ota(sys.argv[1], sys.argv[2]))
        else:
            if not isValidAddress(sys.argv[1]):
                print("Invalid Address: ", sys.argv[1])
            if not path.exists(sys.argv[2]):
                print("File not found: ", sys.argv[2])
    else:
        print("Specify the device address and firmware file")
        print(">python ota.py \"01:23:45:67:89:ab\" \"firmware.bin\"")
